<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MySQL DB Inspector</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background-color: #f4f4f4; color: #333; }
        .container { background-color: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
        h1, h2, h3 { color: #333; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input[type="text"], input[type="password"] { width: calc(100% - 22px); padding: 10px; margin-bottom: 15px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        input[type="submit"] { background-color: #007bff; color: white; padding: 10px 15px; border: none; border-radius: 4px; cursor: pointer; font-size: 16px; }
        input[type="submit"]:hover { background-color: #0056b3; }
        .error { color: red; font-weight: bold; margin-bottom: 15px; }
        .success { color: green; font-weight: bold; margin-bottom: 15px; }
        table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f0f0f0; }
        .table-container { margin-top: 20px; }
        pre { background-color: #eee; padding: 10px; border: 1px solid #ccc; border-radius: 4px; white-space: pre-wrap; word-wrap: break-word; }
        .edit-form-container { margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee; }
        .action-buttons input[type="submit"] { background-color: #28a745; margin-right: 10px; }
        .action-buttons input[type="submit"]:hover { background-color: #218838; }
        .action-buttons a { display: inline-block; padding: 10px 15px; background-color: #6c757d; color: white; text-decoration: none; border-radius: 4px; font-size: 16px; }
        .action-buttons a:hover { background-color: #5a6268; }
        textarea { width: calc(100% - 22px); padding: 10px; margin-bottom: 15px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; min-height: 80px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>MySQL Database Inspector</h1>
        <form method="POST" action="b.php">
            <label for="db_host">Database Host:</label>
            <input type="text" id="db_host" name="db_host" value="<?php echo $db_host_form; ?>" required>

            <label for="db_user">Database Username:</label>
            <input type="text" id="db_user" name="db_user" value="<?php echo $db_user_form; ?>" required>

            <label for="db_pass">Database Password:</label>
            <input type="password" id="db_pass" name="db_pass" value="<?php echo $db_pass_form; ?>">

            <label for="db_name">Database Name:</label>
            <input type="text" id="db_name" name="db_name" value="<?php echo $db_name_form; ?>" required>

            <input type="submit" name="submit" value="İncele">
        </form>

        <?php
        // Report all PHP errors (useful for debugging)
        error_reporting(E_ALL);
        ini_set('display_errors', 1);

        function getPrimaryKeys($conn, $tableName) {
            $pks = [];
            $sql = "SHOW KEYS FROM \`" . $tableName . "\` WHERE Key_name = 'PRIMARY'"; // Removed real_escape_string
            $result = $conn->query($sql);
            if ($result && $result->num_rows > 0) {
                while ($row = $result->fetch_assoc()) {
                    $pks[] = $row['Column_name'];
                }
                $result->free();
            }
            return $pks;
        }

        function getColumnMetadata($conn, $tableName) {
            $columns = [];
            $sql = "SHOW COLUMNS FROM \`" . $tableName . "\`"; // Removed real_escape_string
            $result = $conn->query($sql);
            if ($result && $result->num_rows > 0) {
                while ($row = $result->fetch_assoc()) {
                    $columns[$row['Field']] = $row; // Store full metadata including Type
                }
                $result->free();
            }
            return $columns;
        }

        $db_host_form = isset($_REQUEST['db_host']) ? htmlspecialchars($_REQUEST['db_host']) : 'localhost';
        $db_user_form = isset($_REQUEST['db_user']) ? htmlspecialchars($_REQUEST['db_user']) : '';
        $db_pass_form = isset($_REQUEST['db_pass']) ? htmlspecialchars($_REQUEST['db_pass']) : '';
        $db_name_form = isset($_REQUEST['db_name']) ? htmlspecialchars($_REQUEST['db_name']) : '';

        // Handle Update Action
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_data') {
            $db_host = $_POST['db_host'];
            $db_user = $_POST['db_user'];
            $db_pass = $_POST['db_pass'];
            $db_name = $_POST['db_name'];
            $table_name = $_POST['table_name'];
            
            $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);
            if ($conn->connect_error) {
                echo "<p class='error'>Connection failed for update: " . htmlspecialchars($conn->connect_error) . "</p>";
            } else {
                $primaryKeys = getPrimaryKeys($conn, $table_name);
                $update_fields = [];
                $where_conditions = [];
                $columnMetadata = getColumnMetadata($conn, $table_name);

                foreach ($_POST['fields'] as $field_name => $field_value) {
                    $update_fields[] = "\`" . $conn->real_escape_string($field_name) . "\` = '" . $conn->real_escape_string($field_value) . "'";
                }

                foreach ($primaryKeys as $pk_name) {
                    if (isset($_POST['original_pk'][$pk_name])) {
                        $where_conditions[] = "\`" . $conn->real_escape_string($pk_name) . "\` = '" . $conn->real_escape_string($_POST['original_pk'][$pk_name]) . "'";
                    } else {
                        echo "<p class='error'>Primary key " . htmlspecialchars($pk_name) . " not found in submission for update.</p>";
                        $conn->close();
                        exit;
                    }
                }

                if (!empty($update_fields) && !empty($where_conditions)) {
                    $sql_update = "UPDATE \`" . $conn->real_escape_string($table_name) . "\` SET " . implode(', ', $update_fields) . " WHERE " . implode(' AND ', $where_conditions);
                    if ($conn->query($sql_update) === TRUE) {
                        echo "<p class='success'>Record updated successfully in table " . htmlspecialchars($table_name) . ".</p>";
                    } else {
                        echo "<p class='error'>Error updating record: " . htmlspecialchars($conn->error) . "</p>";
                        echo "<p>Query: " . htmlspecialchars($sql_update) . "</p>";
                    }
                } else {
                    echo "<p class='error'>No fields to update or primary key missing for table " . htmlspecialchars($table_name) . ".</p>";
                }
                $conn->close();
            }
            // Link to go back to the main view, preserving DB connection parameters
            $back_params = http_build_query([
                'db_host' => $db_host,
                'db_user' => $db_user,
                'db_pass' => $db_pass, // Be cautious with passing password in GET, though it's for a local tool
                'db_name' => $db_name,
                'submit' => 'İncele' // To trigger the main display
            ]);
            echo "<p><a href='b.php?{$back_params}'>Back to Inspector</a></p>";

        // Handle Edit Form Display Action
        } elseif ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'edit_row') {
            $db_host = $_GET['db_host'];
            $db_user = $_GET['db_user'];
            $db_pass = $_GET['db_pass']; // Caution: password in GET
            $db_name = $_GET['db_name'];
            $table_name = $_GET['table_name'];

            $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);
            if ($conn->connect_error) {
                echo "<p class='error'>Connection failed for edit: " . htmlspecialchars($conn->connect_error) . "</p>";
            } else {
                $primaryKeys = getPrimaryKeys($conn, $table_name);
                $columnMetadata = getColumnMetadata($conn, $table_name);
                $where_conditions = [];
                $original_pk_values = [];

                foreach ($primaryKeys as $pk_name) {
                    if (isset($_GET[$pk_name])) {
                        $pk_value = $_GET[$pk_name];
                        $where_conditions[] = "\`" . $conn->real_escape_string($pk_name) . "\` = '" . $conn->real_escape_string($pk_value) . "'";
                        $original_pk_values[$pk_name] = $pk_value;
                    } else {
                        echo "<p class='error'>Primary key " . htmlspecialchars($pk_name) . " not provided for editing.</p>";
                        $conn->close();
                        exit;
                    }
                }

                if (!empty($where_conditions)) {
                    $sql_fetch_row = "SELECT * FROM \`" . $conn->real_escape_string($table_name) . "\` WHERE " . implode(' AND ', $where_conditions) . " LIMIT 1";
                    $result_row = $conn->query($sql_fetch_row);

                    if ($result_row && $result_row->num_rows > 0) {
                        $row_to_edit = $result_row->fetch_assoc();
                        echo "<div class='edit-form-container'>";
                        echo "<h2>Edit Row in Table: " . htmlspecialchars($table_name) . "</h2>";
                        echo "<form method='POST' action='b.php'>";
                        echo "<input type='hidden' name='action' value='update_data'>";
                        echo "<input type='hidden' name='db_host' value='" . htmlspecialchars($db_host) . "'>";
                        echo "<input type='hidden' name='db_user' value='" . htmlspecialchars($db_user) . "'>";
                        echo "<input type='hidden' name='db_pass' value='" . htmlspecialchars($db_pass) . "'>"; // Caution
                        echo "<input type='hidden' name='db_name' value='" . htmlspecialchars($db_name) . "'>";
                        echo "<input type='hidden' name='table_name' value='" . htmlspecialchars($table_name) . "'>";

                        foreach ($primaryKeys as $pk_name) {
                            echo "<input type='hidden' name='original_pk[" . htmlspecialchars($pk_name) . "]' value='" . htmlspecialchars($original_pk_values[$pk_name]) . "'>";
                        }

                        foreach ($row_to_edit as $field_name => $field_value) {
                            $column_type_full = isset($columnMetadata[$field_name]['Type']) ? strtolower($columnMetadata[$field_name]['Type']) : '';
                            $is_pk = in_array($field_name, $primaryKeys);
                            
                            echo "<label for='field_" . htmlspecialchars($field_name) . "'>" . htmlspecialchars($field_name) . " (" . htmlspecialchars($column_type_full) . ($is_pk ? " - PRIMARY KEY" : "") . "):</label>";
                            
                            if (strpos($column_type_full, 'text') !== false || strpos($column_type_full, 'blob') !== false || strpos($column_type_full, 'varchar') !== false && intval(preg_replace('/[^0-9]/', '', $column_type_full)) > 255 ) {
                                echo "<textarea id='field_" . htmlspecialchars($field_name) . "' name='fields[" . htmlspecialchars($field_name) . "]'>" . htmlspecialchars($field_value) . "</textarea><br>";
                            } else {
                                echo "<input type='text' id='field_" . htmlspecialchars($field_name) . "' name='fields[" . htmlspecialchars($field_name) . "]' value='" . htmlspecialchars($field_value) . "'" . ($is_pk ? " readonly" : "") . "><br>";
                            }
                        }
                        echo "<div class='action-buttons'>";
                        echo "<input type='submit' value='Save Changes'>";
                        $back_params = http_build_query([
                            'db_host' => $db_host, 'db_user' => $db_user, 'db_pass' => $db_pass, 'db_name' => $db_name, 'submit' => 'İncele'
                        ]);
                        echo "<a href='b.php?{$back_params}'>Cancel</a>";
                        echo "</div>";
                        echo "</form>";
                        echo "</div>";
                        $result_row->free();
                    } else {
                        echo "<p class='error'>Row not found for editing or error fetching row: " . htmlspecialchars($conn->error) . "</p>";
                         echo "<p>Query: " . htmlspecialchars($sql_fetch_row) . "</p>";
                    }
                } else {
                     echo "<p class='error'>Primary key information missing for edit action.</p>";
                }
                $conn->close();
            }
        // Handle Main Display Action ("İncele")
        } elseif (isset($_POST['submit'])) {
            $db_host = $db_host_form; // Use the sanitized versions from top
            $db_user = $db_user_form;
            $db_pass = $db_pass_form;
            $db_name = $db_name_form;

            // Create connection
            $conn = new mysqli($db_host, $db_user, $db_pass, $db_name);

            // Check connection
            if ($conn->connect_error) {
                echo "<p class='error'>Connection failed: " . htmlspecialchars($conn->connect_error) . "</p>";
            } else {
                echo "<p class='success'>Successfully connected to the database '" . htmlspecialchars($db_name) . "' on host '" . htmlspecialchars($db_host) . "'.</p>";

                // 1. Get and display tables
                $sql_tables = "SHOW TABLES";
                $result_tables = $conn->query($sql_tables);

                if ($result_tables === false) {
                    echo "<p class='error'>Error fetching tables: " . htmlspecialchars($conn->error) . "</p>";
                } elseif ($result_tables->num_rows > 0) {
                    echo "<div class='table-container'>";
                    echo "<h2>Tables in '" . htmlspecialchars($db_name) . "':</h2>";
                    echo "<ul>";
                    $tables = [];
                    while ($row_table = $result_tables->fetch_array()) {
                        $table_name = $row_table[0];
                        $tables[] = $table_name;
                        echo "<li>" . htmlspecialchars($table_name) . "</li>";
                    }
                    echo "</ul>";
                    echo "</div>";

                    // 2. For each table, get columns and data
                    foreach ($tables as $table_name) {
                        echo "<div class='table-container'>";
                        echo "<h3>Table: " . htmlspecialchars($table_name) . "</h3>";

                        // Get columns
                        $sql_columns = "SHOW COLUMNS FROM \`" . $conn->real_escape_string($table_name) . "\`";
                        $result_columns = $conn->query($sql_columns);

                        if ($result_columns === false) {
                            echo "<p class='error'>Error fetching columns for table " . htmlspecialchars($table_name) . ": " . htmlspecialchars($conn->error) . "</p>";
                        } elseif ($result_columns->num_rows > 0) {
                            echo "<h4>Columns:</h4>";
                            echo "<table><tr>";
                            $column_names = [];
                            while ($row_column = $result_columns->fetch_assoc()) {
                                echo "<th>" . htmlspecialchars($row_column['Field']) . " (" . htmlspecialchars($row_column['Type']) . ")</th>";
                                $column_names[] = $row_column['Field'];
                            }
                            echo "</tr></table>";

                            // Get data (dump)
                            $sql_data = "SELECT * FROM \`" . $conn->real_escape_string($table_name) . "\`";
                            $result_data = $conn->query($sql_data);

                            if ($result_data === false) {
                                echo "<p class='error'>Error fetching data for table " . htmlspecialchars($table_name) . ": " . htmlspecialchars($conn->error) . "</p>";
                            } elseif ($result_data->num_rows > 0) {
                                echo "<h4>Data Dump (first 100 rows):</h4>";
                                echo "<table>";
                                // Table header
                                echo "<tr>";
                                foreach ($column_names as $col_name) {
                                    echo "<th>" . htmlspecialchars($col_name) . "</th>";
                                }
                                echo "<th>Actions</th>"; // Add Actions header
                                echo "</tr>";

                                // Table rows
                                $rowCount = 0;
                                while ($row_data = $result_data->fetch_assoc()) {
                                    if ($rowCount >= 100) { // Limit to 100 rows for performance
                                        echo "<tr><td colspan='" . count($column_names) . "'>... and more rows (display limited to 100).</td></tr>";
                                        break;
                                    }
                                    echo "<tr>";
                                    foreach ($column_names as $col_name) {
                                        echo "<td>" . htmlspecialchars($row_data[$col_name]) . "</td>";
                                    }
                                    // Construct Edit Link
                                    $primaryKeys = getPrimaryKeys($conn, $table_name);
                                    $edit_link_params = [
                                        'action' => 'edit_row',
                                        'table_name' => $table_name,
                                        'db_host' => $db_host,
                                        'db_user' => $db_user,
                                        'db_pass' => $db_pass, // Caution with password in GET
                                        'db_name' => $db_name
                                    ];
                                    $pk_valid_for_link = true;
                                    foreach ($primaryKeys as $pk_name) {
                                        if (isset($row_data[$pk_name])) {
                                            $edit_link_params[$pk_name] = $row_data[$pk_name];
                                        } else {
                                            $pk_valid_for_link = false; // Should not happen if PKs are correct
                                            break;
                                        }
                                    }
                                    if ($pk_valid_for_link && !empty($primaryKeys)) {
                                        echo "<td><a href='b.php?" . http_build_query($edit_link_params) . "'>Edit</a></td>";
                                    } else if (empty($primaryKeys)) {
                                        echo "<td>No PK</td>"; // No primary key defined for table
                                    } else {
                                        echo "<td>PK Error</td>"; // PK value missing in data
                                    }
                                    echo "</tr>";
                                    $rowCount++;
                                }
                                echo "</table>";
                            } else {
                                echo "<p>No data in table '" . htmlspecialchars($table_name) . "'.</p>";
                            }
                            if ($result_data) $result_data->free();
                        } else {
                            echo "<p>No columns found for table '" . htmlspecialchars($table_name) . "'.</p>";
                        }
                        if ($result_columns) $result_columns->free();
                        echo "</div>"; // end table-container for this table
                    }
                } else {
                    echo "<p>No tables found in the database '" . htmlspecialchars($db_name) . "'.</p>";
                }
                if ($result_tables) $result_tables->free();
                $conn->close();
            }
        }
        ?>
    </div>
</body>
</html>