import os
import requests
from urllib.parse import urlparse, urlunparse

def check_admin_path(url_to_check):
    """
    Checks if the /admin/ path is accessible for a given base URL.
    Returns True if accessible (HTTP 200), False otherwise.
    """
    try:
        # Ensure the URL has a scheme, default to http if not present
        parsed_url = urlparse(url_to_check)
        if not parsed_url.scheme:
            # Reconstruct with http if no scheme, then re-parse
            # This handles cases like "example.com"
            parsed_url = urlparse(f"http://{url_to_check}")

        # Construct the admin URL from scheme and netloc only
        admin_url = urlunparse((parsed_url.scheme, parsed_url.netloc, 'admin/', '', '', ''))

        headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
        }
        # Allow redirects and verify SSL, but set a timeout
        response = requests.get(admin_url, headers=headers, timeout=10, allow_redirects=True, verify=True)
        
        print(f"  Checking: {admin_url} -> Status: {response.status_code}")
        return response.status_code == 200
    except requests.exceptions.Timeout:
        print(f"  Timeout while checking: {admin_url}")
        return False
    except requests.exceptions.ConnectionError:
        print(f"  Connection error for: {admin_url}")
        return False
    except requests.exceptions.RequestException as e:
        print(f"  Error checking {admin_url}: {e}")
        return False
    except Exception as e:
        print(f"  An unexpected error occurred with {admin_url}: {e}")
        return False

def main():
    input_filename = "site.txt"
    output_filename = "sahip.txt"

    if not os.path.exists(input_filename):
        print(f"Error: Input file '{input_filename}' not found.")
        return

    print(f"Reading URLs from '{input_filename}'...")
    found_admin_urls = []

    with open(input_filename, 'r', encoding='utf-8') as infile:
        for line in infile:
            original_url = line.strip()
            if not original_url:  # Skip empty lines
                continue
            
            print(f"\nProcessing URL: {original_url}")
            if check_admin_path(original_url):
                found_admin_urls.append(original_url)
                print(f"  SUCCESS: Admin path found for {original_url}")
            else:
                print(f"  INFO: Admin path not found or not accessible for {original_url}")

    if found_admin_urls:
        print(f"\nWriting {len(found_admin_urls)} URLs with accessible admin paths to '{output_filename}'...")
        with open(output_filename, 'w', encoding='utf-8') as outfile:
            for url in found_admin_urls:
                outfile.write(url + "\n")
        print("Done.")
    else:
        print("\nNo URLs with accessible admin paths were found.")

if __name__ == "__main__":
    main()